# Info:
# Script requires R="4.3.1" and ggplot2="3.4.3" to function properly (you can use for example r-lib rig to change your default R version: https://github.com/r-lib/rig)
# Replace "path/to/.../" with the according path to the input-file or output directory
# feed supplementary tables 1 and 2 to the script, which can be found here: https://osf.io/n5qj6/

# On Ubuntu you need to install following library first: libgdal-dev & libfontconfig1-dev
# sudo apt install libgdal-dev
# sudo apt install libfontconfig1-dev

# Install "remotes"-package for versionized installation of necessary packages
install.packages("remotes")

# Install necessary packages with specific versions
remotes::install_version("ggplot2", version = "3.4.3")
remotes::install_version("ggnewscale", version = "0.4.10")
remotes::install_version("maptools", version = "1.1-8")
remotes::install_version("rgeos", version = "0.6-4")

# Install necessary packages
install.packages("dplyr")
install.packages("cowplot")
install.packages("ggmap")
install.packages("ggpolypath")
install.packages("ggrepel")
install.packages("gridExtra")
install.packages("raster")
install.packages("scales")
install.packages("stringr")
install.packages("svglite")

# Load libraries
library(ggplot2)
library(cowplot)
library(dplyr)
library(ggmap)
library(ggnewscale)
library(ggpolypath)
library(ggrepel)
library(gridExtra)
library(maptools)
library(rgeos)
library(raster)
library(scales)
library(stringr)

########################################################################################
# fetch data for "Landkreise"
# If raster::getData is not working you can also load "geoData.RData" from https://osf.io/n5qj6/
DISTRICTS.shp <- raster::getData("GADM", country = "DEU", level = 2)
# transform fetched "Landkreis"-data into dataframe & filter for all Thuringian "Landkreise"
DISTRICTS_DF <- fortify(DISTRICTS.shp, region = "CC_2")
DISTRICTS_DF_THURINGIA <- DISTRICTS_DF[DISTRICTS_DF$id >= "16000",] # Only Thuringian "Landkreise" carry an Id > 16000

# create df
MOBILITY_DF <- read.csv(file = "/path/to/supplementary_table_6_guided_sampling_mobility-datasets.tsv", sep = '\t') %>%
  mutate(Random_Sampling_Oct22 = na_if(Random_Sampling_Oct22, 0)) %>%
  mutate(Random_Sampling_Nov22 = na_if(Random_Sampling_Nov22, 0)) %>%
  mutate(Guided_Sampling = na_if(Guided_Sampling, 0))
MOBILITY_DF$Trips <- ifelse(MOBILITY_DF$Trips_Oct20 >= MOBILITY_DF$Trips_Jun21, MOBILITY_DF$Trips_Oct20, MOBILITY_DF$Trips_Jun21)
MOBILITY_DF$Hit_By <- as.factor(MOBILITY_DF$Hit_By)
MOBILITY_DF$Trip_Group <- ifelse(MOBILITY_DF$Trips > 10000, 1,
                                 ifelse(MOBILITY_DF$Trips <= 10000 & MOBILITY_DF$Trips > 2000, 2,
                                        ifelse(MOBILITY_DF$Trips <= 2000 & MOBILITY_DF$Trips > 0, 3, 4)))
MOBILITY_DF$Trip_Group <- as.factor(MOBILITY_DF$Trip_Group)
MOBILITY_THURINGIA_DF <- filter(MOBILITY_DF, Federal_State == "Thuringia")

SAMPLES_DF <- read.csv(file = "/path/to/supplementary_table_5-guided_sampling_samples.tsv", sep = '\t')
SAMPLES_DF_OCT <- filter(SAMPLES_DF, Isolation_Date < 20221101)
SAMPLES_DF_NOV <- filter(SAMPLES_DF, Isolation_Date >= 20221101 | Searched_Lineage == "first" | (Searched_Lineage == "TRUE" & Isolation_Date < 20221101 & Sampling == "randomized"))
SAMPLES_DF_NOV$Searched_Lineage[which(SAMPLES_DF_NOV$Isolation_Date < 20221026)] <- "first"

GUIDED_SAMPLING_DF <- filter(SAMPLES_DF_OCT, Sampling == "guided" | Sampling == "first")
RANDOMIZED_SAMPLING_DF <- filter(SAMPLES_DF_OCT, Sampling == "randomized" | Sampling == "first")

# plot maps with samples and mobility

GUIDED_SAMPLING_MAP <- ggplot() +
  geom_polypath(
    data = DISTRICTS_DF_THURINGIA,
    aes(x = long, y = lat, group = group),
    color = "#444444",
    fill = "#ffffff",
    size = 0.15
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 3)), #filter(filter(SAMPLES_THURINGIA_DF, !(Hit_By == "first")), !(Random_Sampling_Oct22 == 0))
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#C77AE1',
    shape = 25
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 2)), #filter(filter(SAMPLES_THURINGIA_DF, !(Hit_By == "first")), !(Random_Sampling_Oct22 == 0))
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#8626A6',
    shape = 25
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 1)), #filter(filter(SAMPLES_THURINGIA_DF, !(Hit_By == "first")), !(Random_Sampling_Oct22 == 0))
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#501764',
    shape = 25
  ) +
  scale_fill_manual(
    values = c("3" = '#F1DEF8', "2" = '#C77AE1', "1" = '#8626A6'),
    name = "Trips",
    labels = c("3"  = "< 5000", "2" = "< 10000", "1" = ">= 10000")
  ) +
  geom_point(
    data = filter(GUIDED_SAMPLING_DF, Searched_Lineage == "FALSE"),
    aes(x= longitude, y = latitude,  col = Searched_Lineage, shape = Searched_Lineage),
    position = position_jitter(width = 0.04, height = 0.04, seed = 113),
    alpha = 0.6,
    stroke = 1,
    size = 4
  ) +
  geom_point(
    data = filter(GUIDED_SAMPLING_DF, !(Searched_Lineage == "FALSE")),
    aes(x= longitude, y = latitude,  col = Searched_Lineage, shape = Searched_Lineage),
    stroke = 1,
    size = 4
  ) +
  scale_color_manual(
    values = c("FALSE" = '#0088CC', "TRUE" = '#e33955', "first" = '#F8A63A'),
    name = "Lineage",
    labels = c("Not Found","Found","Origin")
  ) +
  scale_shape_manual(
    values = c("FALSE" = 16, "TRUE" = 16, "first" = 16),
    name = "Lineage",
    labels = c("Not Found","Found","Origin")
  ) +
  geom_label_repel(
    data = filter(GUIDED_SAMPLING_DF, !(Badge_Description == "FALSE")),
    aes(x= longitude, y = latitude, label = Badge_Description, color = Searched_Lineage),
    size = 9,
    box.padding = unit(0.2, "lines"),
    point.padding = unit(0.3, "lines"),
    nudge_x = c("15th Oct 2022" = 1.1, "5th Oct 2022; initial BQ.1.1-sample" = 1.9),
    nudge_y = c("15th Oct 2022" = -0.1, "5th Oct 2022; initial BQ.1.1-sample" = 0.21)
  ) +
  new_scale_color() +
  scale_color_manual(values = c("first" = '#F8A63A', "TRUE" = '#e33955')) +
  guides(
    size = "none",
    alpha = "none"
  ) +
  theme(
    plot.title = element_text(hjust = 0.5, size = 20),
    legend.position = "bottom",
    legend.title = element_blank(),
    legend.text = element_text(size = 14, margin = margin(r = 1.5, unit = "cm")),
    legend.spacing.x = unit(0.1, 'cm'),
    legend.key = element_rect(fill = "white"),
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1
  )
GUIDED_SAMPLING_MAP


## randomized sampling
RANDOMIZED_SAMPLING_OCT_MAP <- ggplot() +
  geom_polypath(
    data = DISTRICTS_DF_THURINGIA,
    aes(x = long, y = lat, group = group),
    color = "#444444",
    fill = "#ffffff",
    size = 0.15
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 3)),
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#D6D6D6',
    shape = 25
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 2)),
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#999999',
    shape = 25
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 1)),
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#666666',
    shape = 25
  ) +
  scale_fill_manual(
    values = c("3" = '#EBEBEB', "2" = '#ADADAD', "1" = '#7A7A7A'),
    name = "Trips",
    labels = c("3"  = "< 5000", "2" = "< 10000", "1" = ">= 10000")
  ) +
  geom_point(
    data = filter(RANDOMIZED_SAMPLING_DF, Searched_Lineage == "FALSE"),
    aes(x= longitude, y = latitude,  col = Searched_Lineage, shape = Searched_Lineage),
    position = position_jitter(width = 0.04, height = 0.04, seed = 56),
    alpha = 0.6,
    stroke = 1,
    size = 4
  ) +
  geom_point(
    data = filter(RANDOMIZED_SAMPLING_DF, !(Searched_Lineage == "FALSE")),
    aes(x= longitude, y = latitude,  col = Searched_Lineage, shape = Searched_Lineage),
    stroke = 1,
    size = 4
  ) +
  scale_color_manual(values = c("FALSE" = '#0AADFF', "TRUE" = '#e33955', "first" = '#F8A63A')) +
  scale_shape_manual(values = c("FALSE" = 16, "TRUE" = 16, "first" = 16)) +
  geom_label_repel(
    data = filter(RANDOMIZED_SAMPLING_DF, !(Badge_Description == "FALSE")),
    aes(x= longitude, y = latitude, label = Badge_Description, color = Searched_Lineage),
    size = 9,
    box.padding = unit(0.2, "lines"),
    point.padding = unit(0.3, "lines"),
    nudge_x = c("5th Oct 2022; initial BQ.1.1-sample" = 1.7),
    nudge_y = c("5th Oct 2022; initial BQ.1.1-sample" = 0.21)
  ) +
  new_scale_color() +
  scale_color_manual(values = c("first" = '#F8A63A', "TRUE" = '#e33955')) +
  new_scale_color() +
  theme(
    plot.title = element_text(hjust = 0.5, size = 20),
    legend.position = "none",
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1
  )
RANDOMIZED_SAMPLING_OCT_MAP
#ggsave("randomized_sampling_map.svg", plot = RANDOMIZED_SAMPLING_MAP, path = "/path/to/output", width = 1100, height = 1000, units = "px", dpi = 300, scale = 1.3)

RANDOMIZED_SAMPLING_NOV_MAP <- ggplot() +
  geom_polypath(
    data = DISTRICTS_DF_THURINGIA,
    aes(x = long, y = lat, group = group),
    color = "#444444",
    fill = "#ffffff",
    size = 0.15
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 3)),
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#D6D6D6',
    shape = 25
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 2)),
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#999999',
    shape = 25
  ) +
  geom_point(
    data = filter(MOBILITY_THURINGIA_DF, (Trip_Group == 1)),
    aes(x= longitude, y = latitude, fill = Trip_Group, stroke = 0.5),
    size = 4,
    color = '#666666',
    shape = 25
  ) +
  scale_fill_manual(
    values = c("3" = '#EBEBEB', "2" = '#ADADAD', "1" = '#7A7A7A'),
    name = "Trips",
    labels = c("3"  = "< 5000", "2" = "< 10000", "1" = ">= 10000")
  ) +
  geom_point(
    data = filter(SAMPLES_DF_NOV, Searched_Lineage == "FALSE"),
    aes(x= longitude, y = latitude,  col = Searched_Lineage, shape = Searched_Lineage, alpha = 0.15),
    position= position_jitter(width = 0.03, height = 0.03, seed = 73),
    #alpha = 0.6,
    stroke = 1,
    size = 4
  ) +
  geom_point(
    data = filter(SAMPLES_DF_NOV, !(Searched_Lineage == "FALSE")),
    aes(x= longitude, y = latitude,  col = Searched_Lineage, shape = Searched_Lineage),
    stroke = 1,
    size = 4
  ) +
  scale_color_manual(values = c("FALSE" = '#0AADFF', "TRUE" = '#e33955', "first" = '#F8A63A')) +
  scale_shape_manual(values = c("FALSE" = 16, "TRUE" = 16, "first" = 16)) +
  geom_label_repel(
    data = filter(SAMPLES_DF_NOV, !(Badge_Description == "FALSE")),
    aes(x= longitude, y = latitude, label = Badge_Description, color = Searched_Lineage),
    size = 9,
    box.padding = unit(0.2, "lines"),
    point.padding = unit(0.3, "lines"),
    nudge_x = c("16th Nov 2022" = 0.17, "5th Oct 2022; initial BQ.1.1-sample" = 1.7, "1st Nov 2022 - Upper" = 0.7, "1st Nov 2022 - lower" = 0.9, "15th Nov 2022" = 0.08),
    nudge_y = c("16th Nov 2022" = -0.15, "5th Oct 2022; initial BQ.1.1-sample" = 0.21, "1st Nov 2022 - Upper" = 0.08, "1st Nov 2022 - Lower" = 0, "15th Nov 2022" = 0.15),
    show.legend = FALSE
  ) +
  new_scale_color() +
  scale_color_manual(values = c("first" = '#F8A63A', "TRUE" = '#e33955', "FALSE" = '#0AADFF'), labels = c("Not Found", "Origin", "Found" )) +
  guides(
    color = guide_legend(position = "bottom", override.aes = list(shape = 1, size = 6, stroke = 1.2)),
    alpha = FALSE,
    shape = FALSE,
    size = FALSE
  ) +
  theme(
    plot.title = element_text(hjust = 0.5, size = 20),
    legend.position = "none",
    panel.background = element_blank(),
    axis.text = element_blank(),
    axis.ticks = element_blank(),
    axis.title = element_blank(),
    aspect.ratio = 1
  )
RANDOMIZED_SAMPLING_NOV_MAP

COMMONG_LEGEND <- get_legend(GUIDED_SAMPLING_MAP)
GUIDED_SAMPLING_MAP <- GUIDED_SAMPLING_MAP +
  theme(legend.position = "none")
GUIDED_SAMPLING_MAP
#ggsave("randomized_sampling_after_map.svg", plot = RANDOMIZED_SAMPLING_AFTER_MAP, path = "/path/to/output", width = 1100, height = 1000, units = "px", dpi = 300, scale = 1.3)

UPPER_ROW <- plot_grid(GUIDED_SAMPLING_MAP, RANDOMIZED_SAMPLING_OCT_MAP, RANDOMIZED_SAMPLING_NOV_MAP, ncol = 3, align = 'h')
UPPER_ROW
COMBINED_MAPS = plot_grid(UPPER_ROW, COMMONG_LEGEND, ncol = 1, rel_heights = c(1, 0.1))
COMBINED_MAPS

ggsave("new_mobility_guided_map.svg", plot = COMBINED_MAPS, path = "/path/to/output", width = 6000, height = 2400, units = "px", dpi = 300, scale = 1.3)
